#ifndef CODEC_H_
#define CODEC_H_

#include <inttypes.h>
#include <Wire.h>


//CODEC FOR AC101, ESP32_A1S_AC101

typedef enum
{
	BM_LR,	//bypass both L and R channel (default)
	BM_L,	//bypass L channel only
	BM_R  	//bypass R channel only
} BYPASS_MODE;

typedef enum
{
  IM_LR,    //input mode = line-in L, line-in R
  IM_LMIC   //input mode = Line-in L, Microphone
} 
INPUT_MODE;

 INPUT_MODE inputMode; 


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

//Codec, the base class
class Codec
{
  protected:
	int i2cAddress;
	virtual bool writeReg(uint8_t reg, uint16_t val){};
	virtual uint16_t readReg(uint8_t reg){};
	
  private:
	int inmode;
	TwoWire ii2c = TwoWire(0);
	uint8_t _pinsda, _pinscl;
	uint32_t __i2cspeed;
	
	bool writeReg(uint8_t reg, uint16_t val);
	uint16_t readReg(uint8_t reg);
	
	// METHODS SPECIAL TO AC101 CODEC
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	bool setI2sWordSize(uint16_t size);
	bool setI2sFormat(uint16_t format);
	bool setI2sMode(uint16_t mode);
	bool setI2sSampleRate(uint16_t rate);
	bool setI2sClock(uint16_t bitClockDiv, uint16_t bitClockInv, uint16_t lrClockDiv, uint16_t lrClockInv);
	
	//leftchannel input selector methods
	bool leftMic1(bool select);		//left channel mic1 select
	bool leftLineDiff(bool select);	//left channel line difference (line Left- Line Right)
	bool leftLineLeft(bool select);	//left channel line (L)

	//left output mixer source select/deselect
	bool omixerLeftLineLeft(bool select);
	bool omixerLeftMic1(bool select);
	bool omixerLeftDacLeft(bool select);

	//right output mixer source select/deselect
	bool omixerRightLineRight(bool select);
	bool omixerRightMic1(bool select);
	bool omixerRightDacRight(bool select);

	//rightchannel input selector methods
	bool rightMic1(bool select);		//right channel mic1 select
	bool rightLineDiff(bool select);	//right channel line difference (line Left- Line Right)
	bool rightLineRight(bool select);	//right channel line (R)
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
  public:
  
    float outCorrectionGain;
	bool* muteLeftAdcIn;
	bool* muteRightAdcIn;
	
	//Constructor method to init the i2c bus at specific pins and clock frequency
	bool Codec(uint8_t _sda, uint8_t _scl, uint32_t _speed);
	
	~Codec();

	//function to detect codec chip wired to the i2c pins, return the i2c address if found
 	identify(int sdaPin, int sclPin, uint32_t frequency);

	uint8_t* readAllReg();
	
	//initialize the codec
	bool init(int address);
	
	//get and set the input mode
	//mode 0: L+R, mode 1: L+MIC
	bool setInputMode(int mode){inmode = mode; return true;};
	int getInputMode(){ return inmode;};
	
	//get and set microphone gain (0:0dB,1-7:30dB-48dB)
	virtual uint8_t getMicGain();
	virtual bool setMicGain(uint8_t gain);
	
	//get and set microphone noise gate (0-31: -76.5dB, -75.0dB,...., -30.0dB)
	virtual int getMicNoiseGate();
	virtual bool setMicNoiseGate(int gate);
	
	//optimize the analog to digital conversion range
	//range: 0, 1, 2, 3, 4 (1Vrms/2.83Vpp, 0.5Vrms/1.41Vpp, 0.25Vrms/707mVpp, 0.125Vrms/354mVpp, 0.625Vrms/177mVpp)
	virtual void optimizeConversion(int range=2);
	
	//get and set the output level (analog gain)
	//vol = 0-31
	bool setOutputVolume(int vol);
	int getOutputVolume();
		
	//get and set the input gain (analog)
	bool setInputGain(int gain){return false;};
	int getInputGain(){return 0;};	
	
	//bypassed the analog input to the output, disconnect the digital i/o 
	virtual bool analogBypass(bool bypass, BYPASS_MODE bm=BM_LR);  

	//bypassed the analog input to the output, disconnect the digital input, preserve the digital output connection
	virtual bool analogSoftBypass(bool bypass, BYPASS_MODE bm=BM_LR);  
	
	// ES8388 methods not here:
	// outputSelect(), inputSelect(), DACmute(), setALCmode(), 
	// mixerSourceSelect(), mixerSourceControl(), setBitsPerSample()
	
};


#endif
